/**
 * External dependencies
 */
import { __, _n, sprintf } from '@wordpress/i18n';
import { SearchListControl } from '@woocommerce/components';
import PropTypes from 'prop-types';

/**
 * Internal dependencies
 */
import { withSearchedProducts } from '../../hocs';

/**
 * The products control exposes a custom selector for searching and selecting
 * products.
 *
 * @param {Object} props Component props.
 * @param {Function} props.onChange  Callback fired when the selected item changes
 * @param {Function} props.onSearch  Callback fired when a search is triggered
 * @param {Array}    props.selected  An array of selected products.
 * @param {Array}    props.products  An array of products to select from.
 * @param {boolean}  props.isLoading Whether or not the products are being loaded.
 *
 * @return {Function} A functional component.
 */
const ProductsControl = ( {
	onChange,
	onSearch,
	selected,
	products,
	isLoading,
} ) => {
	const messages = {
		clear: __( 'Clear all products', 'woocommerce' ),
		list: __( 'Products', 'woocommerce' ),
		noItems: __(
			"Your store doesn't have any products.",
			'woocommerce'
		),
		search: __(
			'Search for products to display',
			'woocommerce'
		),
		selected: ( n ) =>
			sprintf(
				_n(
					'%d product selected',
					'%d products selected',
					n,
					'woocommerce'
				),
				n
			),
		updated: __(
			'Product search results updated.',
			'woocommerce'
		),
	};
	return (
		<SearchListControl
			className="woocommerce-products"
			list={ products }
			isLoading={ isLoading }
			selected={ selected }
			onSearch={ onSearch }
			onChange={ onChange }
			messages={ messages }
		/>
	);
};

ProductsControl.propTypes = {
	onChange: PropTypes.func.isRequired,
	onSearch: PropTypes.func,
	selected: PropTypes.array,
	products: PropTypes.array,
	isLoading: PropTypes.bool,
};

ProductsControl.defaultProps = {
	selected: [],
	products: [],
	isLoading: true,
};

export default withSearchedProducts( ProductsControl );
